//---------------------------------------------------------------
// window.onloadメソッド
//概要:
//	遺伝情報の組み立ての主処理。
//	本関数内で使用している独自に定義したオブジェクトDNAとProteinは
//	本コード後半で定義している。
//引数:
//	なし
//戻り値:
//	なし
//---------------------------------------------------------------
window.onload = function()
	{
		$( '#result' ).append( '<h1>Submission for GenoCon2012 Challenge B</h1>' );
		$( '#result' ).append( '<p class="message">Start!</p>' );
		try
		{
			$( '#result' ).append( '<h2>FDH:</h2>');
			var FDH = new Protein();
			FDH.loadSequence( 'rdf1s327i', 'protein', 'http://www.uniprot.org/uniprot/P46154' );
			FDH.write();
			$( '#result' ).append( '<p>length = ' + FDH.getLength() + ' aa</p>');
			
			$( '#result' ).append( '<h2>cold-adapted FDH:</h2>');
			var coldAdaptedFDH = FDH.substitute( 'G', 319 );
			coldAdaptedFDH.write();
			$( '#result' ).append( '<p>length = ' + coldAdaptedFDH.getLength() + ' aa</p>');
			
			$( '#result' ).append( '<h2>(signal peptide) + (cold-adapted FDH):</h2>');
			var coldAdaptedFDHwithSP = FDH.addSignalPeptide( 'http://scinets.org/item/cria228s2ria228s3i' );
			coldAdaptedFDHwithSP.write();
			$( '#result' ).append( '<p>length = ' + coldAdaptedFDHwithSP.getLength() + ' aa</p>');
			
			
			$( '#result' ).append( '<h2>Result of reverse-translation:</h2>');
			var finalSeq = coldAdaptedFDHwithSP.reverseTranslate( 'rdf1s291i', 'codon_table_of_Arabidopsis_thaliana' );
			finalSeq.write();
			$( '#result' ).append( '<p>length = ' + finalSeq.getLength() + ' bp</p>');
			
			$( '#result' ).append( '<p class="message">successfully-completed!<p>' );			
		}
		catch( e )
		{
			$( '#result' ).append( '<p class="error">' + e + '</p>' );
		}
	};

////////////////////////////////////////////////////////////////////////////////////
//
//	以下は遺伝情報の組み立てのための汎用的モジュール
//
////////////////////////////////////////////////////////////////////////////////////


//---------------------------------------------------------------
//Sequenceクラスのコンストラクタ
//概要:
//	DNA、Proteinなどのシーケンスを格納・管理するための
//	クラスの親クラス。Sequenceクラス自体のインスタンスは生成せず
//	抽象クラスとして扱うものとする。
//引数:
//	なし
//戻り値:
//	なし
//---------------------------------------------------------------
function Sequence()
{
	//Sequenceクラスは抽象クラス
}


//---------------------------------------------------------------
//Sequenceクラスのメンバ _checkSequenceメソッド
//概要:
//	シーケンスに既定のアルファベット(塩基やアミノ酸)以外の文字
//	が含まれていないか検査し、その場合はエラーを出す。
//引数:
//	seq 検査のシーケンス対象となるシーケンスの文字列
//戻り値:
//	なし
//---------------------------------------------------------------
Sequence.prototype._checkSequence = function( seq )
	{
		var i;
		for( i = 0; i < seq.length; i++ )
		{
			var j;
			for( j = 0; j < this.constructor._alphabet.length; j++ )
			{
				if( seq.charAt( i ) === this.constructor._alphabet[ j ] )
				{
					break;
				} 
			}
			if( j === this.constructor._alphabet.length )
			{
				throw new Error( 'シーケンスデータの' + j + '番目に不正なアルファベット' + seq.charAt( i ) + 'が含まれています。' );
			}
		}
	};


//---------------------------------------------------------------
//Sequenceクラスのメンバ _checkPositionメソッド
//概要:
//	残基番号が不正でないか検査。
//引数:
//	position 検査のシーケンス対象となる残基番号
//戻り値:
//	なし
//---------------------------------------------------------------
Sequence.prototype._checkPosition = function( position )
	{
		if( position < 0 )
		{
			throw new Error( '残基番号として' + position + 'が指定されました。1以上の値を指定してください。' );
		}
		if( position > this.getLength() )
		{
			throw new Error( '残基番号として残基数よりも大きい値(' + position + ')が指定されました。' );
		}
	};


//---------------------------------------------------------------
//Sequenceクラスのメンバ setSequenceメソッド
//概要:
//	オブジェクトにDNAやタンパク質のシーケンスをセットする。
//	複数の引数を与えるとそれらの配列を準に連結してセットする。
//引数:
//	arg シーケンスの文字列か、同じクラスのオブジェクト
//戻り値:
//	なし
//---------------------------------------------------------------
Sequence.prototype.setSequence = function( arg )
	{
		var sum = '';
		var i;
		for( i = 0; i < arg.length; i++ )
		{
			if( this.constructor !== arg[ i ].constructor && ( typeof arg[ i ] ) !== 'string' )
			{
				throw new Error( '文字列以外の値あるいは異なるクラスのオブジェクトが引数として与えられました。' );
			}
			sum += arg[ i ].toString().replace( /\s/g, '' );
		}
		this._checkSequence( sum );
		this._sequence = sum;
	};


//---------------------------------------------------------------
//Sequenceクラスのメンバ loadSequenceメソッド
//概要:
//	シーケンスをLinkDataのテーブルデータから取得して、オブジェクトにセットする。
//	基本的にはLinkData.getObjectsのwrapperだが、
//	propertyの#以前の省略に対応。(例: http://linkdata.org/property/rdf1s222i#Sequence → #Sequence)
//	さらにpropertyが省略された場合には、http://linkdata.org/property/[WORK ID]#Sequenceをpropertyとする。
//引数:
//	workID		LinkDataの作品のID
//	filename	作品中のデーブルのファイル名
//	subject		主語 (表の各行の見出しに相当)
//	property	術後 (表の各列の見出しに相当)
//戻り値:
//	なし
//---------------------------------------------------------------
Sequence.prototype.loadSequence = function( workId, filename, subject, property )
	{
		var seq;
		var emp = [];
		if( property === undefined )
		{
			property = '#Sequence';
		}
		if( property.charAt( 0 ) === '#' )
		{
			property = 'http://linkdata.org/property/' + workId + property;
		}
		
		seq = LinkData.getObjects( workId, filename, subject, property );
		if( seq[0] === undefined  )
		{
			throw new Error( 'LinkDataからの配列の取得に失敗しました。' );
		}
		
		this.setSequence( seq );
	};


//---------------------------------------------------------------
//Sequenceクラスのメンバ getSequenceメソッド
//概要:
//	オブジェクトにセットされたシーケンスを取得する
//引数:
//	なし
//戻り値:
//	シーケンスの文字列
//---------------------------------------------------------------
Sequence.prototype.getSequence = function()
	{
		if( this._sequence === undefined )
		{
			throw new Error( 'シーケンスがセットされていないオブジェクトのシーケンスが参照されました。' );
		}
		return this._sequence;
	};


//---------------------------------------------------------------
//Sequenceクラスのメンバ toStringメソッド
//概要:
//	シーケンスを文字列で返す。getSequenceを流用しているだけだが、
//	これによってオブジェクトを文字列的に扱える。
//	var P = new Protein( 'ACDEFG' );
//	alert( P )
//	なし
//引数:
//	なし
//戻り値:
//	シーケンスの文字列
//---------------------------------------------------------------
Sequence.prototype.toString = function()
	{
		return this.getSequence();
	};


//---------------------------------------------------------------
//Sequenceクラスのメンバ writeメソッド
//概要:
//	シーケンスを画面に出力。
//戻り値:
//	なし
//---------------------------------------------------------------
Sequence.prototype.write = function()
	{
		var output = '';
		var i;
		var seq = this.getSequence();
		
		$( '#result' ).append( '<div class="sequence">' + seq + '</div>' );
	};


//---------------------------------------------------------------
//Sequenceクラスのメンバ getLengthメソッド
//概要:
//	シーケンスの長さを返す。
//引数:
//	なし
//戻り値:
//	シーケンスの長さ
//---------------------------------------------------------------
Sequence.prototype.getLength = function()
	{
		return this.getSequence().length;
	};



//---------------------------------------------------------------
//Sequenceクラスのメンバ fragmentateメソッド
//概要:
//	シーケンスの断片のオブジェクトを生成。
//引数:
//	start	断片のになる残基の位置 (1始まり)
//	end		断片の末尾になる残基の位置 (1始まり)
//戻り値:
//	生成された断片のオブジェクト
//---------------------------------------------------------------
Sequence.prototype.fragmentate = function( start, end )
	{
		var seq;
		var len;
		
		this._checkPosition( start );
		this._checkPosition( end );

		if( start > end )
		{
			throw new Error( 'fragmentateメソッドの第1引数(' + start + ')に第2引数(' + end + ')よりも大きな値が指定されました。' );
		}

		seq = this.getSequence();
		len = this.getLength();
		
		/* -1するのは先頭の一文字を0番目でなく1番目として扱うため。 */
		seq = seq.substring( start - 1, end );
		return new this.constructor( seq );
	};


//---------------------------------------------------------------
//Sequenceクラスのメンバ substituteメソッド
//概要:
//	シーケンス中の特定の範囲を置換したオブジェクトを生成する。
//引数:
//	subSeq		置換の結果、シーケンスに導入されるシーケンス。
//	start		置換の開始位置の残基番号
//	end			置換の終了位置の残基番号。
//				省略された場合はstartと同じ値がセットされる。
//				つまり、startで指定された1文字だけを置換する。
//戻り値:
//	生成され置換後のオブジェクト
//---------------------------------------------------------------
Sequence.prototype.substitute = function( subSeq, start, end )
	{
		var seq;
		
		if( end === undefined )
		{
			end = start;
		}
		
		this._checkPosition( start );
		this._checkPosition( end );
		
		if( start > end )
		{
			throw new Error( 'substituteメソッドの第2引数(' + start + ')に第3引数(' + end + ')よりも大きな値が指定されました。' );
		}
		
		subSeq = subSeq.toUpperCase();
		this._checkSequence( subSeq );
		
		seq = this.getSequence();
		seq = seq.substring( 0, start - 1 ) + subSeq + seq.substring( end );
		
		return new this.constructor( seq );
	};



//---------------------------------------------------------------
//Sequenceクラスの子クラス DNAクラスのコンストラクタ
//概要:
//	DNAのシーケンス(塩基配列)を格納・管理するためのクラス
//引数:
//	元になるDNAオブジェクトあるいはシーケンスの文字列(複数指定可)
//戻り値:
//	なし
//---------------------------------------------------------------
function DNA()
{
	this.constructor = DNA;
	
	if( arguments.length > 0 )
	{
		this.setSequence( arguments );
	}
}

// Sequenceを継承
DNA.prototype = new Sequence();

//各ヌクレオチドをDNAを構成するアルファベットとして設定。
DNA._alphabet = [ 'A', 'T', 'G', 'C' ];


//---------------------------------------------------------------
//Sequenceクラスの子クラス Proteinクラスのコンストラクタ
//概要:
//	タンパク質のシーケンス(アミノ酸配列)を格納・管理するためのクラス
//引数:
//	元になるProteinオブジェクトあるいはシーケンスの文字列(複数指定可)
//戻り値:
//	なし
//---------------------------------------------------------------
function Protein()
{
	this.constructor = Protein;
	
	if( arguments.length > 0 )
	{
		this.setSequence( arguments );
	}
}

// Sequenceを継承
Protein.prototype = new Sequence();

//アミノ酸と終止コドンに対応する*をタンパク質を構成するアルファベットとして設定。
Protein._alphabet = [ 'A', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'K',
					'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'Y', 'V', 'W', '*' ];



//---------------------------------------------------------------
//Proteinクラスのメンバ addSignalPeptideメソッド
//概要:
//	シーケンスにシグナルペプチドを不可する。
//引数:
//	subject		signal peptide (http://linkdata.org/work/rdf1s292i/)
//				のテーブルデータsignal_peptideのシグナルペプチドを示す主語。
//	position	挿入位置不定の場合はシグナルペプチドの挿入位置。
//				(何番目のアミノ酸残基の後に挿入するか)
//	シグナルペプチドのシーケンス中に不特定部分XXXを含む場合には、
//	場合は3番目以降)の引数で順にXXX部分に入るシーケンスを指定する。
// 例)
// N末端挿入型シグナル XXXXRLXXXXXHL-の始めのXXXXをAAAA
// 次のXXXXXをPPPPPとして付加する場合:
// HPSPHI.addSignalPeptide( 'http://scinets.org/item/cria228s2ria228s8i', 'AAAA', 'PPPPP' );
// 不特定部位挿入型シグナル -PPKKKRKV-を10残基目のアミノ酸残基の
// 直後に挿入する場合:
// HPSHPI.addSignalPeptide( 'http://scinets.org/item/cria228s2ria228s1i', 10 );
//戻り値:
//	なし
//---------------------------------------------------------------
Protein.prototype.addSignalPeptide = function( subject, position )
	{
		//指定されたシグナルペプチドの読み込み
		var signalPeptide = LinkData.getObjects( 'rdf1s292i', 'signal_peptide', subject,
												'http://linkdata.org/property/rdf1s292i#sequence' ).toString();
		
		//挿入部位の情報を読み込み
		var location = LinkData.getObjects( 'rdf1s292i', 'signal_peptide', subject,
										   'http://linkdata.org/property/rdf1s292i#location' ).toString();
		//シグナルペプチド中のXXX部分の置き換えに用いる引数の開始位置。
		//末端に挿入されるシグナルペプチドなら2番目以降の引数が用いられるが、
		//不特定の場合には2番目の引数は挿入位置を示すため、3番目以降になる。
		var start = 2;
		var inputSeq = this.getSequence();
		var outputSeq;
		var i;
		
		//シグナルペプチドから挿入部位を示すハイフンを除去
		signalPeptide = signalPeptide.replace( /-/g , '' );
		switch( location )
		{
			//N末端
			// 元データの綴り間違いと正しい綴りの両方に対応
			case 'N-terminus':
			case 'N-teminus':
				outputSeq = signalPeptide+inputSeq;
				break;
			//C末端
			case 'C-terminus':
				outputSeq = inputSeq+signalPeptide;
				break;
			//不特定
			case 'Any':
				if( position > inputSeq.length || position < 0 )
				{
					throw new Error( 'シグナルペプチドの挿入位置が不正です。(挿入位置:' + position + ')' );
				}
				outputSeq = inputSeq.slice( 0, position ) + signalPeptide + inputSeq.slice( position, inputSeq.length );
				start = 3;
				break;
		}

		//シグナルペプチドに未確定のXを含む場合には、その部分に2番目以降
		//(挿入位置が不特定の場合には3番目)の引数を順番に当てはめていく。
		for( i = start - 1; i < arguments.length; i++ )
		{
			outputSeq = outputSeq.replace( /X+/i , arguments[ i ] );
		}

		return new Protein( outputSeq );
	};


//---------------------------------------------------------------
//Proteinクラスのメンバ _getGeneticCodeTableメソッド
//概要:
//	コドン頻度表を取得し、連想配列に格納。
//引数:
//	workId		LinkData上のコドン頻度表のwork ID
//	filename	LinkData上のコドン表のファイル名
//戻り値:
//	コドン頻度表を格納した連想配列
//---------------------------------------------------------------
Protein.prototype._getGeneticCodeTable = function( workId, filename )
	{
		//遺伝暗号表を格納する連想配列
		var geneticCodeTable = {};
		//一時的にあるアミノ酸に対応するコドンを添字にしてその頻度を格納する連想配列
		var fraction = {};
		var subjects = LinkData.getSubjects( workId, filename );
		var i;
		
		for( i = 0; i < Protein._alphabet.length; i++ )
		{
			var j;
			for( j = 0; j < subjects.length; j++ )
			{
				var aa = LinkData.getObjects( workId, filename, subjects[ j ],
											 'http://linkdata.org/property/' + workId + '#one-letter%20code%20of%20amino%20acid' ).toString();
				
				if( aa === Protein._alphabet[ i ] )
				{
					var codon= subjects[ j ].toString().split( '#' )[1];
					fraction[ codon ] = LinkData.getObjects( workId, filename ,subjects[ j ],
															 'http://linkdata.org/property/' + workId + '#usage' ).toString();
				}
			}
			geneticCodeTable[ Protein._alphabet[ i ] ] = fraction;
			fraction = {};
		}

		return geneticCodeTable;
	};

//---------------------------------------------------------------
//Proteinクラスのメンバ _DHondtメソッド
//概要:
//	比例代表選挙の各政党への議席数の割り振りに用いられる
//	ドント方式を応用して、与えられたコドンの使用頻度から
//	タンパク質の逆翻訳時に使用される各コドンの個数を算出し、
//	その情報を配列に格納して返す。
//引数:
//	totalSeatsNum	タンパク質を構成するあるアミノ酸の総数(総議席数に相当)
//	ratio			各コドンの頻度を含む連想配列(得票数に相当)
//戻り値:
//	コドンを添え字にし、そのコドンの使用数を持つ連想配列
//---------------------------------------------------------------
Protein.prototype._DHondt = function( totalSeatsNum, ratio )
	{
		var max;
		var maxParty;
		var gottenSeats = {};
		var divisor = {};
		var i;
		
		for( i in ratio )
		{
			if( ratio.hasOwnProperty( i ) )
			{
				gottenSeats[ i ] = 0;
				divisor[ i ] = 1;
			}
		}

		for( i = 0; i < totalSeatsNum; i++ )
		{
			var party;
			max = 0;
			for( party in ratio)
			{
				if( ratio.hasOwnProperty( party ) )
				{
					if( max < ( ratio[ party ] / divisor[ party ] ) )
					{
						maxParty = party;
						max = ratio[ maxParty ] / divisor[ maxParty ];
					}
				}
			}
			gottenSeats[ maxParty ]++;
			divisor[ maxParty ]++;
		}
		return gottenSeats;
	};


//---------------------------------------------------------------
//Proteinクラスのメンバ _getAmniAcidCompositionメソッド
//概要:
//	シーケンス中に各アミノ酸が何残基ずつ存在するかを返す。
//引数:
//	inputSeq	対象になるシーケンスの文字列
//戻り値:
//	アミノ酸を添え字にし、そのアミノ酸がシーケンス中で
//	使用された回数を持つ連想配列
//---------------------------------------------------------------
Protein.prototype._getAminoAcidComposition = function( inputSeq )
	{
		var numOfAA = {};
		var i;

		//配列AAnumの初期化
		for( i = 0; i < Protein._alphabet.length; i++ )
		{
			numOfAA[ Protein._alphabet[ i ] ] = 0;
		}

		for( i = 0; i < inputSeq.length; i++ )
		{
			numOfAA[ inputSeq.charAt( i ) ]++;
		}
		return numOfAA;
	};


//---------------------------------------------------------------
//Proteinクラスのメンバ _getIdealCodonCompotitionメソッド
//概要:
//	タンパク質のアミノ酸組成とコドンの使用頻度に基づいて、
//	各コドンの最適な使用回数を算出する。
//引数:
//	geneticCodeTable	頻度情報を含むコドン表のデータ
//	inputSeq			対象になるシーケンスの文字列
//戻り値:
//	アミノ酸を添え字にし、そのアミノ酸の最適な使用回数を持つ連想配列
//---------------------------------------------------------------
Protein.prototype._getIdealCodonCompotition = function( geneticCodeTable, inputSeq )
	{
		var numOfAA = this._getAminoAcidComposition(inputSeq);
		var idealCompo = {};
		var aa;
		for( aa in geneticCodeTable )
		{
			if( geneticCodeTable.hasOwnProperty( aa ) )
			{
				idealCompo[ aa ] = this._DHondt( numOfAA[ aa ], geneticCodeTable[ aa ] );
			}
		}
		return idealCompo;
	};

//---------------------------------------------------------------
//Proteinクラスのメンバ _getIdealCodonCompotitionメソッド
//概要:
//	タンパク質を逆翻訳し、結果をDNAオブジェクトで返す。
//引数:
//	workId		LinkData上のコドン頻度表のwork ID
//	filename	LinkData上のコドン表のファイル名
//戻り値:
//	逆翻訳結果のシーケンスが格納されたDNAオブジェクト
//---------------------------------------------------------------
Protein.prototype.reverseTranslate = function( workId, filename )
	{
		//遺伝暗号表の取得
		var geneticCodeTable = this._getGeneticCodeTable( workId, filename );
		//終止コドン付加
		var inputSeq = this.getSequence() + '*';
		//最適なコドンの使用頻度の取得
		var idealComposition = this._getIdealCodonCompotition( geneticCodeTable, inputSeq );
		//あるアミノ酸に対応するコドンを一時的に格納する配列
		var codons = [];
		//あるアミノ酸ごとに使用するコドンを使用する個数だけ使用する順に詰めたキュー配列
		var tQueue = [];
		//アミノ酸ごとにtQueueの内容を格納したキュー配列
		var codonQueue = {};
		//コドンを使用頻度順にソートするための関数
		var sortCodons = function( a, b ) { return b.number - a.number; };
		//塩基配列を書き出す先
		var outputSeq = '';
		var aa;

		for( aa in idealComposition )
		{
			if( idealComposition.hasOwnProperty( aa ) )
			{
				var codon;
				var i;
				for( codon in idealComposition[ aa ] )
				{
					if( idealComposition[ aa ].hasOwnProperty( codon ) )
					{
						//コドンとその頻度を配列に格納
						codons.push( { 'codon' : codon, 'number' : idealComposition[ aa ][ codon ] } );
					}
				}
				//使用頻度順にコドンをソート
				codons.sort( sortCodons );

				//tQueueに使用する順にコドンを詰め込む
				for( i = 0; i < codons.length; i++ )
				{
					var j;
					for( j = 0; j < codons[ i ].number; j++ )
					{
						tQueue.push( codons[ i ].codon );
					}
				}
				codonQueue[ aa ] = tQueue;
				codons = [];
				tQueue = [];
			}
		}

		//実際にコドンを並べていく。
		for( i = 0; i < inputSeq.length; i++ )
		{
			outputSeq += codonQueue[ inputSeq.charAt( i ) ][ 0 ];
			codonQueue[ inputSeq.charAt( i ) ].shift();
		}
		return new DNA( outputSeq );
	};


