if( typeof( Application ) === 'undefined' || !Application )
{
	var Application = {};
}

Application.chronology = function() {
	this._init.apply( this, arguments );
};

// アプリ全体を1つのオブジェクトして扱う。
Application.chronology.prototype = {
	_workId : null,
	_fileName : null,
	_uriBase : 'http://linkdata.org/',
	_prpDescription: 'http://purl.org/dc/elements/1.1/description',
	_prpBase : null,
	_longAgoFlg : null,
	_calEvents : [],
	_realEvents : {},
	_intervalId : null,
	_colorPalette : [
		'midnightblue', 'seagreen', 'sandybrown', 'indianred', 'indianred', 'dimgray', 'royalblue', 'darkgreen', 'chocolate', 'darkred', 'crimson'
	],
	
	// アプリ起動時にのみ起動する
	_init : function() {
		this._showTblSelecter();
		this._update();
	},
	
	// 年表のセレクタの初期処理
	_showTblSelecter : function() {
		var self = this;
		var selected = 'selected';
		
		$.each( LinkData.getWorks(), function( i, workId ) {
			$.each( LinkData.getFiles( workId ), function( j, fileName ) {
				var sbjBase = self._uriBase + 'resource/' + workId + '#';
				var prpBase = self._uriBase + 'property/' + workId + '#';
				var sbjHead = sbjBase + encodeURIComponent( '[header]' );
				var prpName = prpBase + encodeURIComponent(  'event_name' );
				var titleArr = LinkData.getObjects( workId, fileName, sbjHead, prpName );
				
				if( titleArr.length === 0 )
				{
					//ヘッダー行が見つからない場合は、年表ファイルではないとみなす。
					return true;
				}

				var title = titleArr[ 0 ];
				var tableKey = workId + '|' + fileName;
				$( '#tableSelect' ).append( '<option value="' + tableKey + '" ' + selected + ' >' + title + '</option>' );
				if( selected === 'selected' )
				{
					self._workId = workId;
					self._fileName = fileName;
					selected = '';
				}
			} );
		} );

		$( '#tableSelect' ).change( function() {
			var selected =$( 'option:selected', $( this ) ).val();
			var splitedKey = selected.split( '|' );
			
			self._workId = splitedKey[ 0 ];
			self._fileName = splitedKey[ 1 ];
			self._update();
		});
	},
	
	// 新しい年表が選択される度に呼び出される処理
	_update : function() {
		var self = this;
		
		/* まずはタイマー停止 */
		if( this.intervalId !== undefined )
		{
			clearInterval( this.intervalId );
		}
		
		// URIの生成
		var prpBase = this._uriBase + 'property/' + this._workId + '#';
		this._prpBase = prpBase;
		var sbjBase = this._uriBase + 'resource/' + this._workId + '#';
		this._sbjHead =sbjBase + encodeURIComponent( '[header]' );
		this._prpStart = prpBase + encodeURIComponent( 'start' );
		this._prpEnd = prpBase + encodeURIComponent( 'end' );
		this._prpName = prpBase + encodeURIComponent( 'event_name' );
		
		var now = new Date();
		// 今年の始まりと終わりの時刻を取得
		this._calHead = new Date( now.getFullYear(), 0,1);
		this._calTail = new Date( now.getFullYear() + 1, 0,1 );
		
		// 今年の日数を取得
		this._yearDay = ( this._calTail - this._calHead ) / 1000/60/60/24;
		// LinkData.org上の年表を取得し、配列に格納
		this._createEventTbl();
		// 現在を示すイベントを追加
		this._addNowEvent();
		// Dateオブジェクトで処理可能な範囲の年表か判定し、フラグを立てる。
		this._checkLongAgo();
		// 時間の圧縮処理
		this._shrinkTimeScale();
		// 年表の情報を表示
		this._showInfo();
		// 現在時刻を年表上の時刻に変換する時計をセット
		this._intervalId = setInterval( function() { self._showClock(); }, 100 );
		// カレンダーの表示
		this._showCalendar();
		// イベント一覧のセレクトボックスを表示
		this._showEventSelecter();
		// 処理が終わったので変更された領域を表示する。
		// 初期状態ではこの領域を隠し、このタイミングで表示することで、ロード直後にまだ出来上がっていない状態の見苦しい領域が表示されるのを防ぐ。
		$( '#allArea' ).show();
		$( '#calendar' ).fullCalendar( 'render' );
	},
	
	// 年表を読み取って一旦配列に格納する。
	_createEventTbl : function() {
		var self = this;
		
		self._realEvents = [];
		$.each( LinkData.getSubjects( self._workId, self._fileName ) , function( i, subject ) {
			if( subject === self._sbjHead )
			{
				//これらは期間を指定するためのものなので、イベントとは見なさない。
				return true;
			}
			
			// 年表上のイベント1件に相当する連想配列
			var realEvent = {};
			
			realEvent.title = LinkData.getObjects( self._workId, self._fileName, subject, self._prpName )[ 0 ];
			realEvent.subject = subject;
			realEvent.start = LinkData.getObjects( self._workId, self._fileName, subject, self._prpStart )[ 0 ];
			realEvent.end = LinkData.getObjects( self._workId, self._fileName, subject, self._prpEnd )[ 0 ];
			realEvent.description = LinkData.getObjects( self._workId, self._fileName, subject, self._prpDescription )[ 0 ];
		
			self._realEvents.push( realEvent );
		});
	},
	
	// 年表にはないが、現在を示すイベントを追加
	_addNowEvent : function() {
		var realEvent = {};
		var now = new Date();
		
		realEvent.title = '現在';
		realEvent.id = '[this_time]';
		realEvent.start = $.fullCalendar.formatDate( now, 'yyyy/M/d' );
		realEvent.description = '現実の世界の今は、この歴史圧縮カレンダーではこの日に相当します。';
		
		this._realEvents.push( realEvent );
	},
	
	// 年表の日付がDateオブジェクトで処理できる範囲かを判定する
	_checkLongAgo : function() {
		try{
			// ヘッダーの開始年の読み込み
			var startEraStr = LinkData.getObjects( this._workId, this._fileName, this._sbjHead, this._prpStart )[0];
			if( startEraStr === undefined )
			{
				throw new Error( '年表ファイルのヘッダを認識することができません。' );
			}
		}
		catch( e )
		{
			// 万一、ヘッダが読み取れない場合はエラーとする。
			alert( e );
			return;
		}
		
		// 紀元前28万年以前のデータはDateオブジェクトでは扱えない
		if( parseFloat( startEraStr.split( '/' )[ 0 ] ) < -280000 )
		{
			this._longAgoFlg = true;
		}
		else
		{
			this._longAgoFlg = false;
		}
	},

		// 時間の圧縮を行う。(ただし、実際にはフラグによる分岐のみで下請けに丸投げ)
	_shrinkTimeScale : function() {
		this._calEvents = [];
		if( this._longAgoFlg === true )
		{
			this._shrinkTimeScaleLong();
		}
		else
		{
			this._shrinkTimeScaleShort();
		}
	},

	// Dateオブジェクトで扱える範囲の時間圧縮処理
	_shrinkTimeScaleShort : function() {
		var self = this;
		
		// Dateオブジェクトを対象とした時間圧縮関数
		var shrinkDate = function( date  ) {
			var fromEraHead = date - startEra.getTime();
			return new Date( self._calHead.getTime() + fromEraHead / self._factor );
		};
		
		// 年表全体の開始時刻を取得
		var startTime = new Application.Time( LinkData.getObjects( this._workId, this._fileName, this._sbjHead, this._prpStart )[ 0 ] );
		var startEra = startTime.getMiddleDate();
		this._eraHead = startTime;
		
		// 年表全体の終了時刻を取得
		var endTime = new Application.Time( LinkData.getObjects( this._workId, this._fileName, this._sbjHead, this._prpEnd )[ 0 ] );
		var endEra = endTime.getMiddleDate();
		this._eraTail = endTime;
		
		//年表範囲の期間長
		var periodEra = endEra - startEra;	
		// 時間の倍率 (ミリ秒で計算)
		this._factor =  periodEra / ( this._yearDay * 1000 * 60 * 60 * 24 );
		
		// 各イベントを処理するループ
		$.each( self._realEvents , function( i, realEvent ) {
			// カレンダー上のイベントオブジェクト
			var calEvent = {};
			
			calEvent.title = realEvent.title;
			calEvent.id = realEvent.subject;
			calEvent.allDay =  false;
			calEvent.description = realEvent.description;
			// イベントの背景色はパレットから輪番で使用
			calEvent.color = self._colorPalette[ self._calEvents.length % self._colorPalette.length ];
			
			// --- 開始時刻の処理 ---
			var startTime = new Application.Time( realEvent.start );
			var eventStart = startTime.getMiddleDate();
			
			// 年表上の開始時刻をセット
			calEvent.startReal = startTime;
			// 圧縮後時刻をセット
			calEvent.start = shrinkDate( eventStart );
			// 年表の情報の曖昧さに伴う誤差をセット
			calEvent.startErr = calEvent.start - shrinkDate( startTime.getFirstDate() );
			if( eventStart - startEra === 0 || eventStart - endEra === 0 )
			{
				// 開始時刻ジャストのイベントは定義により誤差はないものとする。
				calEvent.startErr = undefined;
			}
			
			// --- 終了時刻の処理 ---
			var endTime = new Application.Time( realEvent.end );
			if( endTime.year !== undefined )
			{
				calEvent.endReal = endTime;
				var eventEnd = endTime.getMiddleDate();
				calEvent.end = shrinkDate( eventEnd );
				calEvent.endErr = calEvent.end - shrinkDate( endTime.getFirstDate() );
				if( eventEnd - endEra === 0 || eventEnd - startEra === 0 )
				{
					calEvent.endErr = undefined;
				}
			}
			
			self._calEvents.push( calEvent );
		});
	},

	// Dateオブジェクトで扱える範囲外の時間圧縮処理
	_shrinkTimeScaleLong : function() {
		var self = this;
		
		// 時間の圧縮処理
		var shrinkDate = function( year ) {
			var fromEraHead = year - startEra;
			return new Date( self._calHead.getTime() + ( fromEraHead / self._factor )  * 1000 * 60 * 60 * 24 * self._yearDay );
		};
		
		var getYear = function( str ) {
			if( str === undefined )
			{
				return undefined;
			}
			var arr = str.split( '/' );
			return parseFloat( arr[ 0 ], 10 );
		};

		// 年表全体の開始時刻を取得
		var startEra = new Application.LongTime( LinkData.getObjects( this._workId, this._fileName, this._sbjHead, this._prpStart )[ 0 ] );
		this._eraHead = startEra;
		// 年表全体の終了時刻を取得
		var endEra = new Application.LongTime( LinkData.getObjects( this._workId, this._fileName, this._sbjHead, this._prpEnd )[ 0 ] );
		this._eraTail = endEra;
		// 年表範囲の期間長
		var periodEra = endEra - startEra;	

		// 圧縮の倍率をセット
		this._factor = periodEra;

		$.each( self._realEvents, function( i, realEvent ) {
			if( realEvent.subject === self._sbjHead )
			{
				//これらは期間を指定するためのものなので、イベントとは見なさない。
				return true;
			}
			
			// カレンダー上のイベントオブジェクト
			var calEvent = {};
			
			// カレンダー上のイベントのタイトルを設定
			
			calEvent.title = realEvent.title;
			calEvent.id = realEvent.subject;
			calEvent.allDay =  false;
			calEvent.description = realEvent.description;
			// イベントの背景色はパレットから輪番で使用
			calEvent.color = self._colorPalette[ self._calEvents.length % self._colorPalette.length ];

			// --- 開始時刻の処理 ---
			var eventStart = new Application.LongTime( realEvent.start );
			calEvent.startReal = eventStart;
			calEvent.start = shrinkDate( eventStart );
			
			// --- 終了時刻の処理 ---
			var eventEnd = new Application.LongTime( realEvent.end );
			if( eventEnd.getYear() !== undefined )
			{
				calEvent.endReal = eventEnd;
				calEvent.end = shrinkDate( eventEnd );
			}
			
			self._calEvents.push( calEvent );
		});
	},

	// 年表の情報を表示
	_showInfo : function() {
		var sb = [];
		
		// 期間長の文字列を取得
		var getPeriod = function( num ) {
			var oku, man;
			var str;
			if( ( oku = num / 100000000 ) > 1 )
			{
				str = Math.round( oku ) + '億'; 
			}
			else if( ( man = num / 10000 ) > 1 )
			{
				str = Math.round( man ) + '万';
			}
			else
			{
				str = Math.round( num ).toString() ;
			}
			return str;
		};
		
		sb[ sb.length ] = '<h1>年表情報</h1>';
		sb[ sb.length ] = '<h2>年表名</h2>';
		sb[ sb.length ] = LinkData.getObjects( this._workId, this._fileName, this._sbjHead, this._prpName );
		sb[ sb.length ] = '<h2>年表範囲</h2>';
		sb[ sb.length ] = '<p>' + this._eraHead.toString() + '～' + this._eraTail.toString();
		sb[ sb.length ] = '&nbsp（' + getPeriod( this._factor ) + '年間）</p>';
		var description = LinkData.getObjects( this._workId, this._fileName, this._sbjHead, this._prpDescription );
		if( description.length !== 0 )
		{
			sb[ sb.length ] = '<h2>説明</h2>';
			sb[ sb.length ] = '<p>' + description + '</p>';
		}

		$( '#infoArea' ).html( sb.join( '' ) );
	},

	// カレンダー上の現在時刻が年表上のいつなのかを示す時計を表示
	_showClock : function() {
		var sb = [];
		var now = $.fullCalendar.formatDate( new Date(), 'M月d日H時m分ss秒' );
		
		sb[ sb.length ] = '<p>カレンダー上の現在時刻<span class="clock">' + now + '</span>は、';
		sb[ sb.length ] = '年表の<span class="clock">' + this._getExpandedDate( new Date() ) + '</span>に相当します。</p>';
		
		$( '#clockArea' ).html( sb.join( '' ) );
	},
	
	// カレンダー上の時刻を年表上の時刻に膨張させる
	_getExpandedDate : function( date ) {
		var fromCalHead;
		var rtn;
		
		// 0詰め処理のための関数
		var zeroFill = function( num ) {
			var str = num.toString(); 
			if( str.length === 1 )
			{
				str = '0' + str;
			}
			
			return str;	
		};
		

		if( this._longAgoFlg )
		{
			fromCalHead = date - this._calHead;
			rtn = new Application.LongTime( this._eraHead + fromCalHead * this._factor / 1000 / 60 / 60/ 24 / this._yearDay ).toString();
		}
		else
		{
			fromCalHead = date - this._calHead;
			var expDate = new Date( this._eraHead.getMiddleDate().getTime() + fromCalHead * this._factor );
			rtn = $.fullCalendar.formatDate( expDate, 'yyyy年M月d日 H時mm分' );
		}
		return rtn;
	},
	
	// カレンダー表示
	_showCalendar : function()
	{
		var self = this;
		$( '#calendar' ).html( '' );
		$( '#calendar' ).fullCalendar( {
			header : {
				left : 'title',
				right : 'prev,next today'
			},
			titleFormat : {
				month : 'yyyy年 M月'
			},
			dayNames : [ '日曜日', '月曜日', '火曜日', '水曜日', '木曜日', '金曜日', '土曜日' ],
			dayNamesShort : [ '日', '月', '火', '水', '木', '金', '土' ],
			buttonText : {
				today : '今日へジャンプ'
			},
			timeFormat :  '',
			events : self._calEvents,
			eventClick : self._eventClick,
			// こうすることで他の関数と同じく、thisでアプリ自体のオブジェクトを参照できるようにする。
			// さもないとクリックされた日のDateオブジェクトがthisになってしまうのが、元々の仕様。
			dayClick : $.proxy( self._dayClick, self ),
			aspectRatio: 2.5
		});	
	},
	
	// (イベントではなく)カレンダー上の日付がクリックされた場合の処理
	_dayClick : function( date ) {
		var dayHead = new Date( date.getFullYear(), date.getMonth(), date.getDate(), 0, 0, 0 );
		var dayTail = new Date( dayHead.getTime() + 1000 * 60 * 60 * 24 - 1 );
		var sb = [];

		sb[ sb.length ] = '<p>年表上の' + this._getExpandedDate( dayHead );
		sb[ sb.length ] = 'から' + this._getExpandedDate( dayTail ) + 'に相当。</p>'; 
		
		$( '#dialog' ).html( sb.join( '' ) );
		$( '#dialog' ).dialog( {
			autoOpen : true,
			closeOnEscape : true,
			buttons: {
				'閉じる' : function() {
					$(this).dialog( 'close' );
				}
			},
			width: '40%',
			resizable : false,
			title : $.fullCalendar.formatDate( date, 'M月d日' )
		} );
	},

	// イベントがクリックされた場合の処理
	_eventClick : function( calEvent ) {
		var getCalendarDate = function( date, error) {
			var str = $.fullCalendar.formatDate( date, 'M月d日H時m分s秒' );
			
			if( error === undefined )
			{
				return str;
			}
			
			// 以下で情報の曖昧さに伴う誤差の表示
			var year, day, hour, minute, second;
			if( ( year = error / 1000 / 60 / 60 / 24 / 365.2425 ) > 1 )
			{
				str += ' ± ' + Math.round( year ) + '年';
			}
			else if( ( day = year * 365.25 ) > 1 )
			{
				str += ' ± ' + Math.round(day) + '日';
			}
			else if( ( hour = day * 24 ) > 1 )
			{
				str += ' ± ' + Math.round( hour ) + '時間';
			}
			else if( ( minute = hour * 60) > 1 )
			{
				str += ' ± ' + Math.round( minute ) + '分';
			}
			else if( ( second = minute * 60 ) > 1 )
			{
				str += ' ± ' + Math.round( second ) + '秒';
			}
			return str;
		};
		
		var sb = [];
		sb[ sb.length ] = '<h3>カレンダー上の時間</h3>';
		sb[ sb.length ] = '<div>' + getCalendarDate( calEvent.start, calEvent.startErr ) + '</div>' ;
		if( calEvent.end !== null )
		{
			sb[ sb.length ] = '<div>～' + getCalendarDate( calEvent.end, calEvent.endErr ) + '</div>';
		}
		
		sb[ sb.length ] = '<h3>年表上の時間</h3>';
		sb[ sb.length ] = '<div>' + calEvent.startReal.toString() + '</div>';
		if( calEvent.endReal !== undefined )
		{
			sb[ sb.length ] = '<div>～' + calEvent.endReal.toString() + '</div>';
		}
		
		if( calEvent.description !== undefined )
		{
			sb[ sb.length ] = '<h3>解説</h3>';
			sb[ sb.length ] = '<p>' + calEvent.description + '</p>';
		}

		
		$( '#dialog' ).html( sb.join( '' ) );
		$( '#dialog' ).dialog( {
			autoOpen : true,
			closeOnEscape : true,
			buttons: {
				'閉じる' : function() {
					$( this ).dialog( 'close' );
				}
			},
			resizable : false,
			width: '50%',
			title : calEvent.title
		} );
		
	},
	
	// イベントセレクタの表示
	_showEventSelecter : function() {
		var self = this;
		
		$( '#eventSelect' ).html( '' );
		$.each( this._calEvents, function( i, event ) {
			$( '#eventSelect' ).append( '<option value="' + i + '">' +event.startReal.toString() + ' : ' + event.title + '</option>' );
		} );
		
		$( '#eventSelect' ).change( function() {
			var selected =parseInt( $( 'option:selected', $( this ) ).val(), 10 );
			var date = self._calEvents[ selected ].start;
			$( '#calendar' ).fullCalendar( 'gotoDate', date.getFullYear(), date.getMonth(), date.getDate() );
		});
	}
};

// 曖昧さを考慮して時間を扱うためのオブジェクト
Application.Time = function(  ) {
	this._init.apply( this, arguments );
};

Application.Time.prototype = {
	year : undefined,
	month : undefined,
	day : undefined,
	hour : undefined,
	minute : undefined,
	second : undefined,
	msec : undefined,
	
	// コンストラクタ
	_init : function( arg ) {
		if( arg === undefined )
		{
			return;
		}
		
		if( typeof( arg ) === 'string' )
		{
			// 引数がStringならそこからオブジェクト生成
			this.setFromStr( arg );
		}
		else
		{
			//そうでなければ、年月日時間分秒、ミリ秒を個別に設定する。
			this.set.apply( this, arguments );
		}
	},
	
	// 個別の時間要素をオブジェクトにセット
	set: function( year, month, day, hour, minute, second, msec ) {
		this.year = year;
		this.month = month;
		this.day = day;
		this.hour = hour;
		this.minute = minute;
		this.second = second;
		this.msec = msec;
	},
	
	//文字列から時刻をオブジェクトにセット
	setFromStr : function( str ) {
			var arr = str.split( '/' );
			var year = parseInt( arr[ 0 ], 10 );
			var month = parseInt( arr[ 1 ], 10 );
			var day = parseInt( arr[ 2 ], 10 );
			this.year = isNaN( year ) ? undefined : year;
			this.month = isNaN( month ) ? undefined : month;
			this.day = isNaN( day ) ? undefined : day;
	},
	
	// 曖昧さを考慮した場合にありうる最初の時刻を保持するDateオブジェクトを生成
	getFirstDate : function() {
		var first = new Date( this.year, this.month === undefined ? 0 : this.month - 1,
							this.day === undefined ? 1 : this.day,
							this.hour === undefined ? 0 : this.hour,
							this.minute === undefined ? 0 : this.minure,
							this.second === undefined ? 0 : this.second,
							this.msec === undefined ? 0 : this.msec );
		return first;
	},
	
	// 曖昧さを考慮した場合にありうる中央の時刻を保持するDateオブジェクトを生成
	getMiddleDate : function() {
		var middle = new Date();
		middle.setTime( Math.round(( this.getFirstDate().getTime()+ this.getLastDate().getTime() ) / 2 ));
		return middle;
	},
	
	// 曖昧さを考慮した場合にありうる最後の時刻を保持するDateオブジェクトを生成
	getLastDate : function() {
		var last = new Date();
		if( this.msec !== undefined )
		{
			last.setTime( new Date( this.year, this.month - 1, this.day, this.hour, this.minute, this.second, this.msec ) );
		}
		else if( this.second !== undefined )
		{
			last.setTime( new Date( this.year, this.month - 1, this.day, this.hour, this.minute, this.second + 1, 0 ) - 1 );
		}
		else if( this.minute !== undefined )
		{
			last.setTime( new Date( this.year, this.month - 1, this.day, this.hour, this.minute + 1, 0, 0 ) - 1 );
		}
		else if( this.hour !== undefined )
		{
			last.setTime( new Date( this.year, this.month - 1, this.day, this.hour + 1, 0, 0, 0 ) - 1 );
		}		
		else if( this.day !== undefined )
		{
			last.setTime( new Date( this.year, this.month - 1, this.day + 1, 0, 0, 0, 0 ) - 1 );
		}
		else if( this.month !== undefined )
		{
			last.setTime( new Date( this.year, this.month - 1 + 1, 1, 0, 0, 0, 0 ) - 1 );
		}
		else
		{
			last.setTime( new Date( this.year + 1, 0, 1, 0, 0, 0, 0 ) - 1 );
		}
		return last;
	},
	
	toString : function() {
		var str = '';
		if( this.year !== undefined )
		{
			str += this.year + '年';
			if( this.month !== undefined )
			{
				str += this.month + '月';
				if( this.day !== undefined )
				{
					str += this.day + '日';
					if( this.hour !== undefined )
					{
						str += this.hour + '時';
					}
				}
			}
		}
		return str;
	}
};

// 紀元前25万年以前の時刻を扱うためのオブジェクト
Application.LongTime = function(  ) {
	this._init.apply( this, arguments );
};

Application.LongTime.prototype = {
	_year: undefined,
	
	// コンストラクタ
	_init : function( arg ) {
		// 引数として与えられたオブジェクトの型に応じて適切な関数でインスタンス生成
		switch( typeof( arg ) )
		{
			case 'undefined':
				break;
			case 'string':
				this.setFromStr( arg );
				break;
			case 'number':
				this.setYear( arg );
				break;
		}

	},
	
	//文字列から時刻をオブジェクトにセット
	setFromStr : function( str ) {
		this._year = parseFloat( str.split( '/' )[ 0 ] );
	},
	
	// 年をゲット
	getYear : function() {
		return this._year;
	},
	
	// 年をセット
	setYear : function( year ) {
		this._year = year;
	},
	
	// 文字列化
	toString : function() {
		var year = Math.abs( this.getYear() );
		var oku, man, mu;
		var str = '';

		// 読みやすいように漢字で表現
		if( ( oku = Math.floor( year / 100000000 ) ) > 1 )
		{
			str += oku + '億'; 
			year -= oku * 100000000;
		}
		if( ( man = Math.floor( year / 10000 ) ) > 1 )
		{
			str +=  man + '万';
			year -= man * 10000;
		}
		if( ( mu = Math.floor( year ) ) > 1 )
		{
			str += mu.toString();
		}
		if( this.getYear() >= 0 )
		{
			str = '西暦' + str + '年';
		}
		else
		{
			str = '紀元前' + str + '年';
		}
		
		return str;
	},
	
	//数値化
	valueOf : function() {
		return this.getYear();
	}
};

// ここから処理が開始する。実質的にはアプリのオブジェクトを生成するのみ。
$( document ).ready( function() {
	try
	{
		var app = new Application.chronology();
	}
	catch( e )
	{
		alert(e);
	}
} );
